/* uSim ddt.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#include "stdio.h"
#include "digits.h"
#include <ctype.h>
#include <string.h>

#include "memory.h"
#include "system.h"
#include "ddt.h"

/* ParseChar() parses a specific character. */
static int ParseChar(char **x, char c)
{

	if (*(*x) != c)
		goto error;

	(*x) += 1;

	/* All done, no error, return non-zero */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* ParseWord() parses a hexadecimal word value. */
static int ParseWord(char **x, long *value)
{
	int n;

	if ((n = DigitsToLong(*x, value, 16)) == 0)
		goto error;

	if (!IsShort(*value))
		goto error;

	(*x) += n;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* ParseByte() parses a hexadecimal byte value. */
static int ParseByte(char **x, long *value)
{
	int n;

	if ((n = DigitsToLong(*x, value, 16)) == 0)
		goto error;

	if (!IsChar(*value))
		goto error;

	(*x) += n;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* ParseName() parses a name. */
static int ParseName(char **x, char *name, int sizeofName)
{
	char *nextChar = *x;
	char *n = name;

	if (!sizeofName--)
		goto error;

	if (!isalpha(*nextChar))
		goto error;

	while (*nextChar != 0) {

		if (!isalnum(*nextChar) && !strchr("._", *nextChar))
			break;

		if (!sizeofName--)
			goto error;

		*n++ = *nextChar++;

	}

	*n = 0;

	*x = nextChar;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;
	
}

/* DDT ASSEMBLE: As */
static int DDT_A(char *buffer)
{
	char *b = buffer;
	long start;

	if (!ParseChar(&b, 'A'))
		goto error;
	else if (!ParseWord(&b, &start))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer, "LOAD %oQ", start);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT DISPLAY: D  Ds  Ds,f */
static int DDT_D(char *buffer)
{
	char *b = buffer;
	long start;
	long finish;

	if (!ParseChar(&b, 'D'))
		goto error;
	if (ParseChar(&b, 0))
		sprintf(buffer, "DISPLAY");
	else if (!ParseWord(&b, &start))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "DISPLAY %oQ", start);
	else if (!ParseChar(&b, ','))
		goto error;
	else if (!ParseWord(&b, &finish))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer, "DISPLAY %oQ %oQ", start, finish);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT FILL: Fs,f,c */
static int DDT_F(char *buffer)
{
	char *b = buffer;
	long start;
	long finish;
	long byte;

	if (!ParseChar(&b, 'F'))
		goto error;
	else if (!ParseWord(&b, &start))
		goto error;
	else if (!ParseChar(&b, ','))
		goto error;
	else if (!ParseWord(&b, &finish))
		goto error;
	else if (!ParseChar(&b, ','))
		goto error;
	else if (!ParseByte(&b, &byte))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer, "FILL %oQ %oQ %oQ", start, finish, byte);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT GO: G  Gs  Gs,b  Gs,b,c  G,b  G,b,c */
static int DDT_G(char *buffer)
{
	char *b = buffer;
	long start;
	long break1;
	long break2;

	if (!ParseChar(&b, 'G'))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "GO");
	else if (ParseChar(&b, ',')) {
		if (!ParseWord(&b, &break1))
			goto error;
		else if (ParseChar(&b, 0))
			sprintf(buffer, "GO -B%oQ", break1);
		else if (!ParseChar(&b, ','))
			goto error;
		else if (!ParseWord(&b, &break2))
			goto error;
		else if (!ParseChar(&b, 0))
			goto error;
		else
			sprintf(buffer,
			        "GO -B%oQ -B%oQ",
			        break1,
			        break2);
	}
	else if (!ParseWord(&b, &start))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "GO %oQ", start);
	else if (!ParseChar(&b, ','))
		goto error;
	else if (!ParseWord(&b, &break1))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "GO -B%oQ %oQ", break1, start);
	else if (!ParseChar(&b, ','))
		goto error;
	else if (!ParseWord(&b, &break2))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer,
		        "GO -B%oQ -B%oQ %oQ",
		        break1,
		        break2,
		        start);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT LIST: L  Ls  Ls,f */
static int DDT_L(char *buffer)
{
	char *b = buffer;
	long start;
	long finish;

	if (!ParseChar(&b, 'L'))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "LIST");
	else if (!ParseWord(&b, &start))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "LIST %oQ", start);
	else if (!ParseChar(&b, ','))
		goto error;
	else if (!ParseWord(&b, &finish))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer, "LIST %oQ %oQ", start, finish);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT MOVE: Ms,f,d */
static int DDT_M(char *buffer)
{
	char *b = buffer;
	long start;
	long finish;
	long dest;

	if (!ParseChar(&b, 'M'))
		goto error;
	else if (!ParseWord(&b, &start))
		goto error;
	else if (!ParseChar(&b, ','))
		goto error;
	else if (!ParseWord(&b, &finish))
		goto error;
	else if (!ParseChar(&b, ','))
		goto error;
	else if (!ParseWord(&b, &dest))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer, "MOVE %oQ %oQ %oQ", start, finish, dest);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT SET: Ss */
static int DDT_S(char *buffer)
{
	char *b = buffer;
	long start;

	if (!ParseChar(&b, 'S'))
		goto error;
	else if (!ParseWord(&b, &start))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer,"SET %oQ", start);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT TRACE: T  Tn */
static int DDT_T(char *buffer)
{
	char *b = buffer;
	long count;

	if (!ParseChar(&b, 'T'))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "TRACE");
	else if (!ParseWord(&b, &count))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer, "TRACE %oQ", count);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT UNTRACE: U  Un */
static int DDT_U(char *buffer)
{
	char *b = buffer;
	long count;

	if (!ParseChar(&b, 'U'))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "UNTRACE");
	else if (!ParseWord(&b, &count))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer, "UNTRACE %oQ", count);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT EXAMINE: X  Xr */
static int DDT_X(char *buffer)
{
	char *b = buffer;
	char registerName[4];

	if (!ParseChar(&b, 'X'))
		goto error;
	else if (ParseChar(&b, 0))
		sprintf(buffer, "EXAMINE");
	else if (!ParseName(&b, registerName, sizeof(registerName)))
		goto error;
	else if (!ParseChar(&b, 0))
		goto error;
	else
		sprintf(buffer, "EXAMINE %s", registerName);

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* DDT() converts DDT Commands to Monitor Commands. */
int DDT(char *buffer)
{

	/* Return non-zero if there was no error. */
	switch (*buffer) {
	case 'A': return DDT_A(buffer);
	case 'D': return DDT_D(buffer);
	case 'F': return DDT_F(buffer);
	case 'G': return DDT_G(buffer);
	case 'L': return DDT_L(buffer);
	case 'M': return DDT_M(buffer);
	case 'S': return DDT_S(buffer);
	case 'T': return DDT_T(buffer);
	case 'U': return DDT_U(buffer);
	case 'X': return DDT_X(buffer);
	}

	/* Return zero if there was an error. */
error:
	return 0;

}
