/* printf.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#include <stdarg.h>
#include "printf.h"

static inline int _strlen(char *s)
{
	int length;

	for (length = 0; *s++; length++)
		;

	return length;

}

static inline void *_memchr(void *_s, int c, int n)
{
	char *s = (char *)_s;

	do {
		if (n-- <= 0)
			return 0;
	} while (*s++ != c);

	return (void *)--s;

}

int
	_printf(void *output,
	        int (*putc)(int, void *),
	        const char *format,
	        va_list ap)
{
	unsigned count;
	int isLeftJustify;
	int isForceSign;
	int isAlternate;
	int isZeroPad;
	int isPrecisionSpecified;
	int isShortArg;
	int isLongArg;
	int isLongDoubleArg;
	int fieldWidth;
	int precision;
	char sign;
	unsigned i;
	char buffer[32];
	char *s;
	char *t;
	char *digits;
	unsigned long n;

	for (count = 0; *format; format++) {

		if (*format != '%') {
			count++;
			if ((*putc)(*format & 0xFF, output) == -1)
				goto error;
			continue;
		}

		if (*(++format) == '%') {
			count++;
			if ((*putc)(*format & 0xFF, output) == -1)
				goto error;
			continue;
		}

		isLeftJustify = 0;
		isForceSign = 0;
		sign = 0;
		isAlternate = 0;
		isZeroPad = 0;
		fieldWidth = 0;
		precision = 0;
		isPrecisionSpecified = 0;
		isShortArg = 0;
		isLongArg = 0;
		isLongDoubleArg = 0;
		s = &buffer[sizeof(buffer) - 1];

		for (;; format++) {
			switch (*format) {
			case '-':
				isLeftJustify = 1;
				continue;
			case '+':
				isForceSign = 1;
				continue;
			case ' ':
				sign = ' ';
				continue;
			case '#':
				isAlternate = 1;
				continue;
			case '0':
				isZeroPad = 1;
				continue;
			case 0:
				goto error;
			}
			break;
		}
		if (isLeftJustify)
			isZeroPad = 0;

		if (*format == '*') {
			if ((fieldWidth = va_arg(ap, int)) < 0) {
				isLeftJustify = 1;
				fieldWidth = -fieldWidth;
			}
			format++;
		}
		else {
			while ((*format >= '0') && (*format <= '9'))
				fieldWidth = (10 * fieldWidth) + (*format++ - '0');
		}

		if (*format == '.') {
			if ((*++format) == '*') {
				precision = va_arg(ap, int);
				format++;
			}
			else {
				while ((*format >= '0') && (*format <= '9'))
					precision = (10 * precision) + (*format++ - '0');
			}
			if (precision >= 0)
				isPrecisionSpecified = 1;
		}

		for (;; format++) {
			switch (*format) {
			case 'h':
				isShortArg = 1;
				continue;
			case 'l':
				isLongArg = 1;
				continue;
			case 'L':
				isLongDoubleArg = 1;
				continue;
			case 0:
				goto error;
			}
			break;
		}

		switch (*format) {

		case 'd':
		case 'i':
		case 'u':
			n = isLongArg ?
				va_arg(ap, long) : (unsigned long)va_arg(ap, int);
			if (isShortArg)
				n = (short)n;
			if (*format == 'u')
				sign = 0;
			else {
				if ((long)n < 0) {
					n = -n;
					sign = '-';
				}
				else if (isForceSign)
					sign = '+';
			}
			if (!isPrecisionSpecified) {
				if (isZeroPad) {
					precision = fieldWidth;
					if (sign)
						precision--;
				}
				if (precision < 1)
					precision = 1;
			}
			for (i = 0; n; n /= 10, i++)
				*--s = (char)(n % 10 + '0');
			for (; i < precision; i++)
				*--s = '0';
			if (sign) {
				*--s = sign;
				i++;
			}
			break;

		case 'o':
			n = isLongArg ?
				va_arg(ap, long) : (unsigned long)va_arg(ap, int);
			if (isShortArg)
				n = (unsigned short)n;
			if (!isPrecisionSpecified) {
				if (isZeroPad) {
					precision = fieldWidth;
				}
				if (precision < 1)
					precision = 1;
			}
			for (i = 0; n; n /= 8, i++)
				*--s = (char)(n % 8 + '0');
			if (isAlternate && i && (*s != '0')) {
				*--s = '0';
				i++;
			}
			for (; i < precision; i++)
				*--s = '0';
			break;

		case 'p':
			isPrecisionSpecified = 1;
			isLongArg++;
			precision = 8;
		case 'X':
		case 'x':
			digits = (*format == 'X') ?
				"0123456789ABCDEF" : "0123456789abcdef";
			n = isLongArg ?
				va_arg(ap, long) : (unsigned long)va_arg(ap, int);
			if (isShortArg)
				n = (unsigned short)n;
			if (!isPrecisionSpecified) {
				if (isZeroPad) {
					precision = fieldWidth;
					if (isAlternate)
						precision -= 2;
				}
				if (precision < 1)
					precision = 1;
			}
			for (i = 0; n; n /= 16, i++)
				*--s = digits[n % 16];
			for (; i < precision; i++)
				*--s = '0';
			if (isAlternate) {
				*--s = *format;
				*--s = '0';
				i += 2;
			}
			break;

		case 'c':
			*--s = (char)va_arg(ap, int);
			i = 1;
			break;

		case 's':
			if ((s = va_arg(ap, char *)) == 0)
				s = "<0>";
			if (isAlternate) {
				i = (unsigned char)*s++;
				if (isPrecisionSpecified && (i > precision))
					i = precision;
			}
			else {
				if (!isPrecisionSpecified)
					i = _strlen(s);
				else if ((t = _memchr(s, '\0', precision)) != 0)
					i = t - s;
				else
					i = precision;
			}
			break;

		case 'n':
			if (isShortArg)
				*va_arg(ap, short *) = (short)count;
			else if (isLongArg)
				*va_arg(ap, long *) = count;
			else
				*va_arg(ap, int *) = count;
			continue;

		default:
			goto error;

		}

		if (!isLeftJustify)
			while (i < fieldWidth) {
				count++;
				if ((*putc)(' ', output) == -1)
					goto error;
				fieldWidth--;
			}

		while (i > 0) {
			count++;
			if ((*putc)(*s++ & 0xFF, output) == -1)
				goto error;
			i--;
			fieldWidth--;
		}

		while (fieldWidth > 0) {
			count++;
			if ((*putc)(' ', output) == -1)
				goto error;
			fieldWidth--;
		}

	}

	return count;

error:
	return -1;

}
