/* ring.c
 * Copyright (C) 2000, Tsurishaddai Williamson, tsuri@earthlink.net
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/**********************************************************************/

#include <string.h>
#include "ring.h"

/* RingReset() resets a Ring to initial values. */
RingPtr
	RingReset(RingPtr ringPtr,
	          void *elements,
	          unsigned maxElement,
	          unsigned elementSize)
{

	/* Check the parameters... */
	if (ringPtr == 0)
		goto error;
	if (elements == 0)
		goto error;
	if (maxElement == 0)
		goto error;
	if (elementSize == 0)
		goto error;

	/* Reset the ring. */
	ringPtr->count = 0;
	ringPtr->maxElement = maxElement;
	ringPtr->elementSize = elementSize;
	ringPtr->begin = elements;
	ringPtr->end = &(ringPtr->begin[maxElement * elementSize]);
	ringPtr->remove = ringPtr->begin;
	ringPtr->insert = ringPtr->begin;

	/* All done, no error, return a pointer to the Ring. */
	return ringPtr;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* RingCount() counts the Elements in a Ring. */
unsigned RingCount(RingPtr ringPtr)
{

	/* Return the current ring count. */
	return ringPtr->count;

}

/* RingInsert() inserts an Element into a Ring. */
int RingInsert(RingPtr ringPtr, void *element)
{

	/* Error if the Ring has not been Reset. */
	if (ringPtr->maxElement == 0)
		goto error;

	/* Error if the Ring is full. */
	if (ringPtr->count >= ringPtr->maxElement)
		goto error;

	/* Insert the Element into the Ring. */
	memcpy(ringPtr->insert, element, ringPtr->elementSize);

	/* Advance the insertion pointer. */
	ringPtr->insert = &(ringPtr->insert[ringPtr->elementSize]);
	if (ringPtr->insert == ringPtr->end)
		ringPtr->insert = ringPtr->begin;

	/* Count the insertion. */
	ringPtr->count++;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}

/* RingRemove() removes an Element from a Ring. */
int RingRemove(RingPtr ringPtr, void *element)
{

	/* Error if the Ring has not been Reset. */
	if (ringPtr->maxElement == 0)
		goto error;

	/* Error if the Ring is empty. */
	if (ringPtr->count == 0)
		goto error;

	/* Remove the Element from the Ring. */
	memcpy(element, ringPtr->remove, ringPtr->elementSize);

	/* Advance the remove pointer. */
	ringPtr->remove = &(ringPtr->remove[ringPtr->elementSize]);
	if (ringPtr->remove == ringPtr->end)
		ringPtr->remove = ringPtr->begin;

	/* Count the removal. */
	ringPtr->count--;

	/* All done, no error, return non-zero. */
	return 1;

	/* Return zero if there was an error. */
error:
	return 0;

}
