/* CPCTRANS v1.1 */
/* Marco Vieth, 9.12.1993 (6.12.1993) */

/* transfer CPC-disks to CPCEMU disk-images and vice versa */
/* now with error-report, if disk-access fails */


#include <stdio.h>	/* for printf */
#include <stdlib.h>	/* for malloc */
#include <string.h>	/* for strcpy */
#include <dos.h>	/* for REGPACK */


//#define DEBUG


#define RETRIES 30


#define MAXSEC 18

#define DRV_READ 2
#define DRV_WRITE 3

#define IMG_DISK_INFO 0x100
#define IMG_TRACK_INFO 0x100


typedef unsigned char byte;
typedef unsigned short int  word;



typedef struct
{
  byte tracks;	/* number of tracks 40, 80, ? */
  byte heads;	/* number of heads 1 or 2 */
  word tsize;	/* track-size in bytes (including track-info) */
		/* computed as spt * (0x0080 << bps) +  IMG_DISK_INFO */
} Disk_info;


typedef struct
{
  byte track;	/* track-number */
  byte head;	/* head-number */
  byte sector;	/* sector number (including offset) */
  byte bps;	/* BPS (bytes per sector) */
  byte state1;	/* state 1 errors */
  byte state2;	/* state 2 errors */
  byte unused1;
  byte unused2;
} Sector_info;

typedef struct
{
  byte track;	/* track-number */
  byte head;	/* head-number */
	/* format-track-parameter : */
  byte bps;	/* BPS (bytes per sector) */
  byte spt;     /* SPT (sectors per track) */
  byte gap3;	/* GAP #3 format (0x4E) */
  byte fill;	/* fill byte (0xe5) */
	/* sector-data : */
  Sector_info si[MAXSEC];

  /* not present in track-info : */
  byte snum;	/* actual sector-number 0..17 */
  byte fsec;	/* first sector (sector-offset+1) */
} Track_info;


typedef struct
{
  byte num;
} Drive;


typedef struct
{
  char fname[80];
  FILE *f;

  byte disk_i[IMG_DISK_INFO];
  byte track_i[IMG_TRACK_INFO];

} Image;


static byte sbuff[0x1000];	/* sector buffer for one sector */



/* **************************** */
/*				*/
/*	something		*/
/*				*/
/* **************************** */



static char upcase(char ch)
{
  return ( ((ch>='a') && (ch<='z')) ? (ch - 0x20) : ch );
}



static char *input(void)
{
  static char line[80];

  /* scanf("%s" , line); */
  gets(line);
  return line;
}


/* **************************** */
/*				*/
/*	read/write drive	*/
/*				*/
/* **************************** */


static void drv_error_report(byte err)
{
  printf("Disk error : ");
  switch (err)
  {
    case 0x00 :
      printf("No error\n");
    break;
    case 0x01 :
      printf("Illegal command\n");
    break;
    case 0x02 :
      printf("Wrong sector address-mark\n");
    break;
    case 0x03 :
      printf("Write protected\n");
    break;
    case 0x04 :
      printf("Sector not found\n");
    break;
    case 0x06 :
      printf("No Disk\n");
    break;
    case 0x08 :
      printf("DMA - overflow\n");
    break;
    case 0x09 :
      printf("DMA - 64K boundary overflow\n");
    break;
    case 0x0C :
      printf("Wrong diskette type\n");
    break;
    case 0x10 :
      printf("CRC error during read\n");
    break;
    case 0x20 :
      printf("Controller error\n");
    break;
    case 0x40 :
      printf("Track not found\n");
    break;
    case 0x80 :
      printf("Drive not found\n");
    break;
    default :
      printf("Unknown error !\n");
    break;
  }
}


static int drv_rw_sector(byte command, Drive *dv, Track_info *t, byte *buffer)
{
  word retry;
  struct REGPACK reg;
  byte number_of_sectors = 1;

  retry = RETRIES;
  do
  {

#ifdef DEBUG
    printf("command=%d, drive=%02d, head=%02d, track=%02d, sector=%02X\n",
     command, dv->num, t->head, t->track, (t->si[t->snum]).sector);
#endif

    /* command = 0x02; for read */

    reg.r_ax = (command << 8) | number_of_sectors;
		/* command , number of sectors to read/write */
    reg.r_cx = (t->track << 8) | (t->si[t->snum]).sector;
		/* track, sector-number (with offset) */
    reg.r_dx = (t->head << 8) | dv->num;
		/* head and drive number */
    reg.r_es = FP_SEG(buffer);   /* DMA-segment */
    reg.r_bx = FP_OFF(buffer);   /* DMA-offset */

    intr(0x13 , &reg);

    if ((reg.r_ax & 0xFF00) != 0)	/* error occured */
    {
      drv_error_report(reg.r_ax>>8);
      if (--retry > 0)
      {
	printf("Error Disk-access drive %c\n", dv->num + 'A');
	return 1;	/* 1 = error */
      }
    }
  }  while ((reg.r_ax & 0xFF00) != 0);
#ifdef DEBUG
  printf("ok.\n");
#endif
  return 0;	/* ok */
}



static int drv_read_sector(Drive *dv, Track_info *t, byte *buffer)
{
  return ( drv_rw_sector(DRV_READ, dv, t, buffer) );
}

static int drv_write_sector(Drive *dv, Track_info *t, byte *buffer)
{
  return ( drv_rw_sector(DRV_WRITE, dv, t, buffer) );
}



/* **************************** */
/*				*/
/*	drive -> image		*/
/*				*/
/* **************************** */


static void create_disk_info(byte *s, Disk_info *d)
{
  static char ident[]  = "MV - CPCEMU Disk-File\r\nDisk-Info\r\n";

  memset(s, 0, IMG_DISK_INFO);
  strcpy((char *)s, ident);

  s += 0x30;
  *(s++) = d->tracks;	/* number of tracks */
  *(s++) = d->heads;	/* number of heads */
  *(s++) = (byte)d->tsize;	/* low byte track size */
  *(s++) = (d->tsize >> 8);	/* high byte track size */
}

static void create_track_info(byte *s, Track_info *t)
{
  static char ident[] = "Track-Info\r\n";
  byte sec;

  memset(s, 0, IMG_TRACK_INFO);
  strcpy((char *)s, ident);

  s += 0x10;
  *(s++) = t->track;	/* track number */
  *(s++) = t->head;	/* head number */
  *(s++) = 0;		/* unused */
  *(s++) = 0;		/* unused */
  /* format-track-parameter : */
  *(s++) = t->bps;	/* BPS (bytes per sector) */
  *(s++) = t->spt;	/* SPT (sectors per track) */
  *(s++) = t->gap3;	/* GAP #3 format (0x4E) */
  *(s++) = t->fill;	/* fill byte (0xe5) */

  /* sector-data (sector ID's) */
  for (sec = 0; sec < t->spt; sec++)
  {
    *(s++) = t->track;	/* track number in ID */
    *(s++) = t->head;	/* head number in ID */
    *(s++) = sec + t->fsec;	/* sector number (with offset) in ID */
    *(s++) = t->bps;	/* bps in ID */

    *(s++) = 0;		/* state 1 errors */
    *(s++) = 0;		/* state 2 errors */
    *(s++) = 0;		/* unused */
    *(s++) = 0;		/* unused */
  }
}





static int img_write(FILE *stream, byte *buf, long len)
{
  if (fwrite(buf, sizeof(byte), len, stream) != len)
  {
    fprintf(stderr, "error fwrite()\n");
    return 1;
  }
  return 0;
}

static int drv_to_img(Drive *dv, Disk_info *d, Track_info *t, Image *im)
{
  byte trk, hd, sec;

  create_disk_info(im->disk_i, d);
  if (img_write(im->f, im->disk_i, IMG_DISK_INFO))  return 1;

  for (trk = 0; trk < d->tracks; trk++)
  {
    t->track = trk;
    for (hd = 0; hd < d->heads; hd++)
    {
      t->head = hd;
      printf("\rLoading track %d  head %d ...  ", trk, hd);
      create_track_info(im->track_i, t);
      if (img_write(im->f, im->track_i, IMG_TRACK_INFO))  return 1;

      for (sec = 0; sec < t->spt; sec++)
      {
	t->snum = sec;
	t->si[t->snum].sector = sec + t->fsec;	/* with offset */
	if (drv_read_sector(dv, t, sbuff))  return 1;
	if (img_write(im->f, sbuff, (0x0080 << t->bps)))  return 1;
      }

    }
  }
  return 0;
}


/* **************************** */
/*				*/
/*	image -> drive		*/
/*				*/
/* **************************** */


static int extract_disk_info(byte *s, Disk_info *d)
{
  static char ident[]  = "MV - CPCEMU Disk-File\r\nDisk-Info\r\n";

  if (memcmp(s, ident, 8))  return 1;	/* not equal to "MV - CPC" */

  s += 0x30;
  d->tracks = *(s++);	/* number of tracks */
  d->heads = *(s++);	/* number of heads */
  d->tsize = *(s++);	/* low byte track size */
  d->tsize |= (*(s++) << 8);	/* high byte track size */
  return 0;
}

static int extract_track_info(byte *s, Track_info *t)
{
  static char ident[] = "Track-Info\r\n";


#ifdef DEBUG
  printf("extract_track_info\n");
#endif

  if (memcmp(s, ident, 5))  return 1;	/* not equal to "Track" */

  s += 0x10;
  t->track = *(s++);	/* track number */
  t->head = *(s++);	/* head number */
  s+=2;			/* unused */
  /* format-track-parameter : */
  t->bps = *(s++);	/* BPS (bytes per sector) */
  t->spt = *(s++);	/* SPT (sectors per track) */
  t->gap3 = *(s++);	/* GAP #3 format (0x4E) */
  t->fill = *(s++);	/* fill byte (0xe5) */

  /* sector-data (sector ID's) */
  memcpy(t->si, s, t->spt * sizeof(Sector_info));

  t->fsec = t->si[0].sector;
#ifdef DEBUG
  printf("t->fsec = %02X\n", t->fsec);
#endif
  return 0;
}



static int img_read(FILE *stream, byte *buf, long len)
{
  if (fread(buf, sizeof(byte), len, stream) != len)
  {
    fprintf(stderr, "error fread()\n");
    return 1;
  }
  return 0;
}



static int img_to_drv(Drive *dv, Disk_info *d, Track_info *t, Image *im)
{
  byte trk, hd, sec;

  if (img_read(im->f, im->disk_i, IMG_DISK_INFO))  return 1;
  if (extract_disk_info(im->disk_i, d))  return 1;

  for (trk = 0; trk < d->tracks; trk++)
  {
    for (hd = 0; hd < d->heads; hd++)
    {
      printf("\rSaving track %d  head %d ...  ", trk, hd);
      if (img_read(im->f, im->track_i, IMG_TRACK_INFO))  return 1;
      if (extract_track_info(im->track_i, t))  return 1;
      if ((trk != t->track) || (hd != t->head))  return 1;

      for (sec = 0; sec < t->spt; sec++)
      {
	t->snum = sec;
	if (img_read(im->f, sbuff, (0x0080 << t->bps)))  return 1;
	if (drv_write_sector(dv, t, sbuff))  return 1;
      }

    }
  }
  return 0;
}



/* **************************** */
/*				*/
/*	select a format		*/
/*				*/
/* **************************** */


static void set_f_data(Disk_info *d, Track_info *t)
{
  d->tracks = 40;	/* number of tracks */
  d->heads = 1;		/* number of heads */

  t->bps = 0x02;	/* BPS = 2 for 0x200 bytes */
  t->spt = 9;		/* SPT (sectors per track) */
  t->gap3 = 0x4E;	/* GAP #3 format (0x4E) */
  t->fill = 0xE5;	/* fill byte (0xe5) */

  t->fsec = 0xC1;	/* first sector */
}

static void set_f_vortex(Disk_info *d, Track_info *t)
{
  d->tracks = 80;	/* number of tracks */
  d->heads = 2;		/* number of heads */

  t->bps = 0x02;	/* BPS = 2 for 0x200 bytes */
  t->spt = 9;		/* SPT (sectors per track) */
  t->gap3 = 0x4E;	/* GAP #3 format (0x4E) */
  t->fill = 0xE5;	/* fill byte (0xe5) */

  t->fsec = 0x01;	/* first sector */
}


static void select_format(Disk_info *d, Track_info *t)
{
  char *s;

  d->tsize = 0xffff;	/* computed below */
  t->track = 0xff;	/* track number, changed during transfer */
  t->head = 0xff;	/* head number, changed during transfer */
  t->snum = 0xff;	/* sector number, changed during transfer */

  set_f_data(d, t);	/* default */

  printf("Select format :\n");
  do
  {
    printf("d)ata 9x512x40x1; s)ystem 9x512x40x1; v)ortex 9x512x80x2;");
    printf(" m)odify; o)k\n");
    printf("Your choice: ");
    s = input();
    switch (upcase(s[0]))
    {
      case 'D' :
	set_f_data(d, t);
	printf("DATA-format set.\n");
      break;
      case 'S' :
	set_f_data(d, t);
	t->fsec = 0x41;		/* first sector for system format */
	printf("SYSTEM-format set.\n");
      break;
      case 'V' :
	set_f_vortex(d, t);
	printf("VORTEX-format set.\n");
      break;
      case 'M' :
	printf("Modify: not implemented.\n");
      break;
      case 'O' :
	printf("Ok.\n");
      break;
      default :
	printf("unknown format.\n");
      break;
    }
  } while (upcase(s[0]) != 'O');
  d->tsize = t->spt * (0x0080 << t->bps) + IMG_DISK_INFO;

}

/* **************************** */
/*				*/
/*	   M E N U		*/
/*				*/
/* **************************** */


static void set_drive(Drive *drv)
{
  char *s;

  printf("Drive with disk in it (A..): ");
  s = input();
  s[0] = upcase(s[0]);
  if ((s[0] >= 'A') && (s[0] <= 'E'))
  {
    drv->num = s[0] - 'A';
  }
  printf("Drive set to %c\n", drv->num + 'A');
}

static void set_image_file(Image *img)
{
  char *s;

  printf("New image file (without .dsk) : ");
  s = input();
  strcpy(img->fname, s);
  if (strchr(img->fname, '.') == NULL)	/* dot found ? */
  {
    strcat(img->fname, ".DSK");
  }
  printf("Image file set to %s\n", img->fname);
}

static void copy_drive_image(Drive *drv, Image *img, Disk_info *di, Track_info *ti)
{
  printf("Copy Drive -> Image\n");
  printf("reading from drive %c.\n", drv->num + 'A');
  printf("writing to image file %s\n", img->fname);
  if ( (img->f = fopen(img->fname, "rb")) != NULL)
  {
    fclose(img->f);
    printf("file already exist. Overwrite (y/n) ? ");
    {
      char *s;
      s = input();
      if (upcase(s[0]) != 'Y')  return;
    }
  }
  if ( (img->f = fopen(img->fname, "wb")) == NULL)
  {
    printf("error opening %s\n", img->fname);
    return;
  }

  select_format(di, ti);

  /* copy drive -> img */
  if (drv_to_img(drv, di, ti, img))
  {
    printf("Operation terminated incorrectly.\n");
  }
  else
  {
    printf("transfer ok.\n");
  }
  fclose(img->f);
}


static void copy_image_drive(Drive *drv, Image *img, Disk_info *di, Track_info *ti)
{
  printf("Copy Image -> Drive\n");

  printf("reading from image file %s\n", img->fname);
  if ( (img->f = fopen(img->fname, "rb")) == NULL)
  {
    printf("error opening %s\n", img->fname);
    return;
  }

  printf("writing to drive %c ...\n", drv->num + 'A');

  printf("Insert disk in drive %c and press return ", drv->num + 'A');
  (void)input();

  /* copy img -> drive */
  if (img_to_drv(drv, di, ti, img))
  {
    printf("Operation terminated incorrectly.\n");
  }
  else
  {
    printf("transfer ok.\n");
  }

  fclose(img->f);

}


static Disk_info *di;
static Track_info *ti;
static Drive *drv;
static Image *img;


static void menu(void)
{
  char *s;

  printf("\n\n");
  printf("CPCTRANS v1.1\n");
  printf("(Transfer CPC-disks to CPCEMU disk-images and vice versa)\n\n");
  do
  {
    printf("\n\n");
    printf("1) Set drive\n");
    printf("2) Set Image file\n");
    printf("3) Copy Drive -> Image\n");
    printf("4) Copy Image -> Drive\n");
    printf("5) Quit\n\n");
    printf("Your choice: ");
    s = input();
    switch (s[0])
    {
      case '1' :
	set_drive(drv);
      break;
      case '2' :
	set_image_file(img);
      break;
      case '3' :
	copy_drive_image(drv, img, di, ti);
      break;
      case '4' :
	copy_image_drive(drv, img, di, ti);
      break;
      case '5' :
      break;
      default :
	printf(" only 1-5 : ");
      break;
    }
  } while (s[0] != '5');
}


int main( /* int argc, char *argv */)
{
  if ((drv = malloc(sizeof(Drive))) == NULL)
  {
    fprintf(stderr, "malloc: no memory\n");
    exit(1);
  }
  drv->num = 0;			/* select drive A */
  if ((img = malloc(sizeof(Image))) == NULL)
  {
    fprintf(stderr, "malloc: no memory\n");
    exit(1);
  }
  strcpy(img->fname, "default.dsk");

  if ((di = malloc(sizeof(Disk_info))) == NULL)
  {
    fprintf(stderr, "malloc: no memory\n");
    exit(1);
  }
  if ((ti = malloc(sizeof(Track_info))) == NULL)
  {
    fprintf(stderr, "malloc: no memory\n");
    exit(1);
  }

  menu();
  return 0;
}
/* end of cpctrans.c */
