/*	mem.c

GEMDOS memory management/program load system

Originally written by JSL as noted below.

MODIFICATION HISTORY

	18 Mar 85	SCC	Added x0term() for function 0x00.

	26 Mar 85	JSL	Modified xexec() to search for a file before it
				tries to load it (to update media change info).

	10 Apr 85	SCC	Modified xexec() to handle long return code for
				setjmp() and longjmp().

	11 Apr 85	SCC	Moved code from ixterm() to ixfreset() to close
				all open files.

	12 Apr 85	SCC	Added init of p_xdta.

			EWF	Added use of 'gemerror.h' and specific error
				codes.

	14 Apr 85	SCC	Backed out modification of 11 Apr 85 to
				ixterm().  It was not the correct fix to the
				problem it attempted to solve.  See note in CLI
				about the ^C problem.

				Modified xexec() to return ERROR on 'file not
				found'.

				This was deemed an appropriate error, since
				0 (go) would return the child's return code, &
				3 (nogo) would return the child's base page
				address.

				Added long type to definition of xsetblk().

				Added long type to definition of xmfree().

				Modified xexec() by casting p to (PD *) to avoid
				compiler warning message.

				Modified xexec() to return ERROR if xpgmld()
				fails.

	19 Apr 85	SCC	Modified xsetblk() and xmalloc() to round to
				even word.

	26 Apr 85	SCC	Modified xpgmld() to return ENSMEM if TPA is
				too small to hold text+data+bss, and to return
				EPLFMT if relocation information would modify
				a value outside of the text or data areas.

	24 May 85	SCC	Modified xexec() to return ERROR if memory for
				environment string or base page cannot be allocated.

	28 May 85	SCC	Modified xterm() to make 'rc' parameter unsigned.

				Modified xexec() and xpgmld() to return more
				informative error numbers.

	 3 Jun 85	SCC	OOPS!  The change mentioned above on 19 Apr 85 to
				xmalloc() wasn't there, so I put it in today.

        15 Jul 85 	LTG	Fixed bug in xpgmld() that returned error ENSMEM
                                if there was exactly enough memory.

	24 Jul 85	SCC	Modified xexec().  Default DTA was being set to
				20 (decimal) longs into base page, rather than 20 (hex).

NAMES

	EWF	Eric W. Fleischman
	JSL	Jason S. Loveman
	SCC	Steven C. Cavender

*/

/* memory management/program load */
#include "fs.h"
#include "gemerror.h"

extern FTAB sft[];
extern long trap13(), setjmp();

PD *run;
MD *ffit();

long xmalloc();
long xsetblk();

int supstk[SUPSIZ]; /* common supervisor stack for all processes */

extern long errbuf[3];
long bakbuf[3];
extern MPB pmd;
extern int osmem[];
extern int osmlen;
int *root[20];
int osmptr;

/**/

ncmps(n,s,d)
int n;
char *s,*d;
{
	while (n--)
		if (*s++ != *d++)
			return(0);

	return(1);
}

int *getosm(n)
int n; /* number of words */
{
	int *m;
	if (n > osmlen) return(0);
	osmlen -= n;
	m = &osmem[osmptr];
	osmptr += n;
	return(m);
}

int *xmgetblk(i)
int i;
{
	int w,*m,*q,**r;
	w = i << 3;
	if (*(r = &root[i]))
	{
		m = *r;
		*r = *((int **) m);
	}
	else if (m = getosm(w+1)) *m++ = i;
	if (q=m) for (i=0; i<w; i++) *q++ = 0;
	return(m);
}

xmfreblk(m)
int *m;
{
	int i;
	i = *(m - 1);
	*((int **) m) = root[i];
	root[i] = m;
}

/**/
/*	Function 0x31	p_termres
*/

xtermres(blkln,rc)
int rc;
long blkln;
{
	MD *m,**q;

	xsetblk(0,run,blkln);

	for (m = *(q = &pmd.mp_mal); m ; m = *q)
		if (m->m_own == run)
		{
			*q = m->m_link; /* pouf ! like magic */
			xmfreblk(m);
		}
		else
			q = &m->m_link;

	xterm(rc);
}	

/**/
/*	Function 0x4C	p_term
*/

xterm(rc)
unsigned int rc;
{
	PD *r;

	dojmp(trap13(5,0x102,-1L));

	run = (r = run)->p_parent;
	ixterm(r);
	run->p_dreg[0] = rc;
	gouser();
}

/**/
/*	Function 0x00	p_term0
*/

x0term()							/* SCC  18 Mar 85 */
{								/* SCC  18 Mar 85 */
	xterm(0);						/* SCC  18 Mar 85 */
}								/* SCC  18 Mar 85 */

/**/
/*
	Last modified	SCC	14 Apr 85
*/

ixterm(r)
PD *r;
{
	MD *m,**q;
	int h, i;

	/* check the standard devices */

	for (i = 0; i < NUMSTD; i++) if ((h = r->p_uft[i]) > 0)
		xclose(h);

	/* now check the sft */

	for (i = 0; i < OPNFILES; i++)
		if (r == sft[i].f_own)
			xclose(i+NUMSTD);


	for (m = *(q = &pmd.mp_mal); m ; m = *q)
	{
		if (m->m_own == r)
		{
			*q = m->m_link;
			freeit(m,&pmd);
		}
		else q = &m->m_link;
	}
}

/**/

ptrup(s,cx)
char *s,*cx;
{
	int i;
	for (i=0; *s && (*s != ' '); i++,s++);
	*cx = *s;
	*s = 0;
	return(i);
}
/**/
/*	Function 0x4B	p_exec

	Last modified	SCC	24 May 85
*/

/* flag = 0: load&go, 3:load/nogo, 4:justgo, 5:create psp */
/* load&go(cmdlin,cmdtail), load/nogo(cmdlin,cmdtail), justgo(psp) */
/* create psp - user receives a memory partition */

long
xexec(flg,s,t,v) /* flag (0 = go, 3 = nogo), command file, tail */
char *s,*t,*v; /* command, tail, environment */
int flg;
{	
	long *p;
	char *b,*e;
	int i,h,j,absflg;
	long rc,max;
	MD *m,*env;
	long *spl;
	int *spw;

	if (flg && ((flg < 3) || (flg > 5)))
		return(EINVFN);

	if ((flg == 0) || (flg == 3))
		if (ixsfirst(s,0,0L))
			return(EFILNF);

	xmovs(sizeof(errbuf),errbuf,bakbuf);

	if (rc = setjmp(errbuf))
	{
		if (rc != E_CHNG)
			longjmp(bakbuf,rc);

		 /* we allocated memory in a previous incarnation */

		if (flg != 4)
		{
			freeit(m,&pmd);
			freeit(env,&pmd);
		}

		longjmp(bakbuf,rc);
	}
	/* will we need memory and a psp ? */
	if (flg != 4)
	{ /* get largest memory partition available */
		if (!v)
			v = run->p_env;
		for (e = v, i = 0; ; i++)
			if (!(*e++))
				if (!(*e++))
					break;
				else
					i++;	/* add an extra */

		i += 2;

		if (i & 1)
			i += 1;

		if (!(env = ffit((long) i,&pmd)))
			return(ENSMEM);

		e = env->m_start;

		/* now copy it */

		while (i--)
			*e++ = *v++;

		if ((max = ffit(-1L,&pmd)) < 0x100)	/* room for base page? */
		{
			freeit(env,&pmd);		/* no, return environment */
			return(ENSMEM);			/* (room for program load */
		}					/* is check in xpgmld()). */

		m = ffit(max,&pmd);
		p = m->m_start;
		m->m_own = env->m_own = (((flg == 0) || (flg == 4)) ? (PD *) p : run);
		max = m->m_length;
		p[0] = p;
		p[1] = ((long) p) + max;

	/* now set up PD (psp) */

		for (i = 0, b = p+2; i < 0x100; i++)
			*b++ = 0;

		p[8] = &p[0x20];	/* default p_xdta is p_cmdlin */
		p[11] = env->m_start;

	/* now inherit standard files from me */

		for (i = 0; i < NUMSTD; i++)
			if ((h = run->p_uft[i]) > 0)
				ixforce(i,run->p_uft[i],p);
			else
				p->p_uft[i] = h;

	/* and current directory set */

		for (i = 0; i < 16; i++)
			ixdirdup(i,run->p_curdir[i],p);

	/* and current drive */

		p->p_curdrv = run->p_curdrv;

	/* copy tail */

		b = p + 0x20;

		for (i = 0; (i < 0x7d) && (*t); i++)
			*b++ = *t++;

		*b++ = 0;
		t = p;
	}

	/* for 3 or 0, need to load, supply baspage containing: */
	/* 	tpa limits, filled in with start addrs,lens */

	if ((flg == 0) || (flg == 3)) 
		if (rc = xpgmld(s,t))
		{
			ixterm(t);			
			return(rc);
		}

	if ((flg == 0) || (flg == 4))
	{
		p = t;
		p->p_parent = run;
		spl = p->p_hitpa;
		*--spl = p;
		*--spl = 0; /* bogus retadd */

		 /* 10 regs (40 bytes) of zeroes  */

		for (i = 0; i < 10; i++)
			*--spl = 0;

		*--spl = p->p_tbase; /* text start */
		spw = (int *) spl;
		*--spw = 0; /* startup status reg */
		spl = (long *) spw;
		*--spl = &supstk[SUPSIZ];
		p->p_areg[6-3] = p->p_areg[7-3] = spl;
		p->p_areg[5-3] = p->p_dbase;
		p->p_areg[4-3] = p->p_bbase;
		run = p;

		if (flg != 5)
			gouser();
	}

	/* sub-func 3 and 5 return here */

	return(t);
}

/**/

spushl(p,val)
long *p,val;
{
	*--p = val;
}

/**/
/*
	Last modified	SCC	26 Apr 85
*/

xpgmld(s,lp)
char *s;
long *lp;
{
	register char *r;
	char *bssadr;
	register int n;
	register long cp,i;
	int h,tmp,absflg;
	long *p,lp0,lpx,flen,lx,codbas,relst,n0,bsslen,tpasiz;
	long len[4]; /* text data bss symbols */

	if ((h = lx = xopen(s)) < 0)
		return(lx);

	xread(h,2L,&tmp);

	if (tmp != 0x601a)
		return(EPLFMT);

	xread(h,16L,len);
	xread(h,4L,&lx); /* junk */
	xread(h,4L,&lx); /* text addr */
	xread(h,2L,&absflg); /* relocation flag */
	flen = len[0] + len[1];
	p = lp + 2;
	tpasiz = lp[1] - lp[0] - 0x100;
	lp += 0x40;
	lp0 = lpx = lp;
	codbas = lp0;
	bssadr = (char *) (lp0 + flen);
	bsslen = tpasiz - flen;

	if (bsslen < len[2])	/* might need to include initial stack size */
		return(ENSMEM);

	for (i = 0; i < 3; i++)
	{
		*p++ = lpx;	/* start of code/data/bss */
		*p++ = len[i];  /* length of code/data/bss */
		lpx += len[i];
	}	

	xread(h,flen,lp0); 			/* read the file */

	if (absflg)
		return(E_OK); 		/* let's split */	

	xlseek(flen + len[3] + 0x1c,h,0);	/* position past symbols */
	xread(h,4L,&relst); 			/* start reloc pointer */

	if (!relst)
		goto zrobss;

	cp = codbas + relst;

	if ((cp < codbas) || (cp >= bssadr))
		return(EPLFMT);

	*((long *) cp) += codbas;

	do
	{
		if ((cp < codbas) || (cp >= bssadr))
			return(EPLFMT);

		/* read some relocation stuff */

		n0 = xread(h,bsslen,bssadr);

		for (i = n0,r = bssadr; i ; i--)
		{ /* r points at reloc item, cp is code pointer */
			if (!(n = *r++))
				goto zrobss;

			if ((n &= 0x00ff) == 1)
				cp += 0x0fe;
			else
			{
				cp += n;
				*((long *) cp) += codbas;
			}
		}
	} while (n0 == bsslen);

	/* zero the bss for him */

zrobss:	for (cp = (long) bssadr, i=3; i < bsslen; i += sizeof(long),cp += sizeof(long))
		*((long *) cp) = 0;

	xclose(h); /* lets be good guys */

	return(E_OK);
}

/**/

MD *ffit(amount,mp)
MPB *mp;
long amount;
{
	MD *p,*q,*p1;	/* free list is composed of MD's */
	MD *qsav;
	int maxflg;
	long maxval;

	q = mp->mp_rover;	/* rotating pointer */
	maxval = 0;

	if (!q)
		return(0); /* no way */

	if (amount == -1L)
		maxflg = 1;
	else
		maxflg = 0;

	p = q -> m_link;

	do /* search the list for an MD with enough space */
	{   /* re-circulate in the list */
		if (!p)
			p = (q = (MD *) &mp->mp_mfl)->m_link;
		if ((!maxflg) && (p->m_length >= amount)) /* big enough */
		{ /* take the whole thing */
			if (p->m_length == amount)
				q->m_link = p->m_link;
			else /* break it up */
			{
				if (!(p1=MGET(MD)))
					return(0);

				p1->m_length = p->m_length - amount;
				p1->m_start = p->m_start + amount;
				p1->m_link = p->m_link;
				p->m_length = amount;
				q->m_link = p1;
			}

			p -> m_link = mp->mp_mal;
			mp->mp_mal = p;
			p->m_own = run;
			mp->mp_rover = (q == &mp->mp_mfl ? q->m_link : q);

			return(p);	/* got some */
		}
		else if (p->m_length > maxval)
			maxval = p->m_length;

		p = (q=p) -> m_link;
	} while (q != mp->mp_rover);

	if (maxflg)
		return(maxval);
	else
		return(0);
}

/**/
/*	Function 0x4A	m_shrink

	Last modified	SCC	19 Apr 85
*/

long
xsetblk(n,blk,len)
int n;
char *blk;
long len;
{
	MD *m,*p;

/*
Traverse the list of memory descriptors looking for this block.
*/

	for (p = pmd.mp_mal; p; p = p->m_link)
		if (blk == p->m_start)
			break;

/*
If block address doesn't match any memory descriptor, then abort.
*/

	if (!p)
		return(EIMBA);

/*
If the caller is not shrinking the block size, then abort.
*/

	if (p->m_length < len)
		return(EGSBF);

/*
Always shrink to an even word length.
*/

	if (len & 1)
		len++;

/*
Create a memory descriptor for the freed portion of memory.
*/

	m = MGET(MD);
	m->m_start = p->m_start + len;
	m->m_length = p->m_length - len;
	p->m_length = len;
	freeit(m,&pmd);

	return(E_OK);
}

/**/

freeit(m,mp)
MD *m;
MPB *mp;
{
	MD *p, *q;

	q = 0;

	for (p = mp->mp_mfl; p ; p = (q=p) -> m_link)
		if (m->m_start <= p->m_start)
			break;

	m->m_link = p;

	if (q)
		q->m_link = m;
	else
		mp->mp_mfl = m;

	if (!mp->mp_rover)
		mp->mp_rover = m;

	if (p)
		if (m->m_start + m->m_length == p->m_start)
		{ /* join to higher neighbor */
			m->m_length += p->m_length;
			m->m_link = p->m_link;

			if (p == mp->mp_rover)
				mp->mp_rover = m;

			xmfreblk(p);
		}

	if (q)
		if (q->m_start + q->m_length == m->m_start)
		{ /* join to lower neighbor */
			q->m_length += m->m_length;
			q->m_link = m->m_link;

			if (m == mp->mp_rover)
				mp->mp_rover = q;

			xmfreblk(m);
		}
}

/**/
/*	Function 0x48	m_alloc

	Last modified	SCC	3 Jun 85
*/

long
xmalloc(amount)
long amount;
{
	MD *m;

/*
Round odd-value requests (except -1L) to next higher even number.
*/

	if ((amount != -1L) && (amount & 1))
		amount++;

/*
Pass the request on to the internal routine.  If it was not able to grant the request,
then abort.
*/

	if (!(m = ffit(amount,&pmd)))
		return(0);

/*
If the request was -1L, the internal routine returned the amount of available memory,
rather than a pointer to a memory descriptor.
*/

	if (amount == -1L)
		return(m);

/*
The internal routine returned a pointer to a memory descriptor.  Return its pointer
to the start of the block.
*/

	return(m->m_start);
}

/**/
/*	Function 0x49	m_free
*/

long
xmfree(addr)
long addr;
{
	MD *p,**q;

	for (p = *(q = &pmd.mp_mal); p; p = *(q = &p->m_link))
		if (addr == p->m_start)
			break;

	if (!p)
		return(EIMBA);

	*q = p->m_link;
	freeit(p,&pmd);

	return(E_OK);
}
